/* Copyright (c) 2017-2023 VMware, Inc. All rights reserved. -- VMware Confidential */
import {FileMigrator, SubContentTransformation} from "./fileMigrator";
import {TransformationInfo} from "../transformationInfo";
import {readFileSync, writeFileSync} from "fs";
import * as path from "path";
import {HtmlFileMigrator} from "./htmlFileMigrator";
import {ExtensionsMigrator} from "./extensionsMigrator";

let readdirp = require('readdirp');
let ncp = require('ncp').ncp;

interface FileTransformations {
   filePath: string,
   contentTransformations: SubContentTransformation[]
}

export class ProjectMigrator {
   static configuration: string = (process.platform === "win32")
      ? ".\\..\\resources\\oldToNewJsAPIs.json"
      : "./resources/oldToNewJsAPIs.json";
   projectPath: string;

   constructor(projectPath: string) {
      this.projectPath = projectPath;
   }

   getPath(): string {
      return this.projectPath;
   }

   migrateProject(): void {
      const transformationInfo: TransformationInfo =
         new TransformationInfo(ProjectMigrator.configuration);

      ProjectMigrator.getFileMigratorsFromDir(this.getPath())
         .then((files: FileMigrator[]) => {
            files.forEach((file) => {
               file.migrateFile(transformationInfo);
               writeFileSync(file.getPath(), file.getContent());
            })
         });
   }

   createProjectTransformations(migrationFile: string): void {
      const transformationInfo: TransformationInfo =
         new TransformationInfo(ProjectMigrator.configuration);

      ProjectMigrator.getFileMigratorsFromDir(this.getPath())
         .then((files: FileMigrator[]) => {
            let toBeMigrated: FileTransformations[] = [];
            files.forEach((file) => {
               const transformation: FileTransformations = {
                  contentTransformations: file.getAllFileTransformations(transformationInfo),
                  filePath: file.getPath().replace(this.getPath(), "")
               };
               toBeMigrated.push(transformation);
            });

            const toWrite: string = JSON.stringify(toBeMigrated, null, "\t");
            writeFileSync(migrationFile, toWrite);
         });
   }

   applyProjectTransformations(migrationFile: string): void {
      const fileTransformations: FileTransformations[] =
         JSON.parse(readFileSync(migrationFile).toString());

      fileTransformations.forEach((elem) => {
         const completePath: string = path.join(this.getPath(), elem.filePath);
         const content: string = readFileSync(completePath).toString();
         const file = new FileMigrator(completePath, content);
         file.applyTransformations(elem.contentTransformations);
         writeFileSync(completePath, file.getContent());
      });
   }

   static getFileMigratorsFromDir(projectDir: string): Promise<FileMigrator[]> {
      return new Promise((resolve, reject) => {
         var settings = {
            root: projectDir,
            entryType: 'files',
            fileFilter: ['*.js', '*.ts', '*.html', 'plugin.xml'],
            directoryFilter: ['!.git', '!node_modules', '!target', "!assets",
               "!dist", "!WEB-INF", "!META-INF", "!locales", "!libs"]
         };
         var allFileMigrators: any[] = [];
         readdirp(settings, (fileInfo) => {
               const content: string = readFileSync(fileInfo.fullPath).toString();
               let file: FileMigrator;
               const fileExtension = path.parse(fileInfo.fullPath).ext;
               if (fileExtension === ".html") {
                  file = new HtmlFileMigrator(fileInfo.fullPath, content);
               } else if (fileExtension === ".xml") { // change if additional support for xml files is added
                  file = new ExtensionsMigrator(fileInfo.fullPath, content);
               } else {
                  file = new FileMigrator(fileInfo.fullPath, content);
               }
               allFileMigrators.push(file);
            },
            (err, res) => {
               if (err) {
                  reject(err);
                  throw err;
               }
               resolve(allFileMigrators);
            }
         );
      });
   }

   static copyProject(source: string, destination: string): Promise<void> {
      return new Promise((resolve, reject) => {
         ncp(source, destination,
            (err) => {
               if (err) {
                  console.error(err);
                  reject();
               }
               resolve();
            });
      });
   }
}
